import React, { useState, useEffect, useCallback } from "react";
import {
  Authenticator,
  Button,
  TextField,
  Heading,
  Flex,
  View,
  Divider,
  Table,
  TableHead,
  TableRow,
  TableCell,
  TableBody,
  Image,
  SelectField,
} from "@aws-amplify/ui-react";
import Modal from "react-modal";
import { Amplify } from "aws-amplify";
import "@aws-amplify/ui-react/styles.css";
import { getUrl } from "aws-amplify/storage";
import { uploadData } from "aws-amplify/storage";
import { generateClient } from "aws-amplify/data";
import outputs from "../amplify_outputs.json";
import { getCurrentUser } from "aws-amplify/auth";

// Enable backend interactions.
Amplify.configure(outputs);
const client = generateClient({
  authMode: "userPool",
});

const AppLayout = ({ children }) => (
  <Flex
    className="App"
    justifyContent="center"
    alignItems="center"
    direction="column"
    width="70%"
    margin="0 auto"
  >
    {children}
  </Flex>
);

const ModalViewer = ({ selectedImage, closeModal }) => {
  const isPDF = String(selectedImage)
    .toLowerCase()
    .split("?")[0]
    .endsWith(".pdf");

  return (
    <Modal
      isOpen={!!selectedImage}
      onRequestClose={closeModal}
      contentLabel="Modal Viewer"
      style={{
        content: {
          display: "flex",
          flexDirection: "column",
          alignItems: "center",
          justifyContent: "center",
          maxWidth: "90%",
          maxHeight: "90%",
          margin: "auto",
          position: "relative",
        },
      }}
    >
      <Button
        style={{
          position: "absolute",
          top: "10px",
          right: "10px",
          backgroundColor: "#f44336",
          color: "#fff",
          border: "none",
          padding: "10px",
          cursor: "pointer",
        }}
        onClick={closeModal}
      >
        Close
      </Button>
      {isPDF ? (
        <iframe
          src={selectedImage}
          title="PDF Viewer"
          style={{ width: "100%", height: "80vh", border: "none" }}
          allowFullScreen
        />
      ) : (
        <Image
          src={selectedImage}
          alt="Full-size image"
          style={{ maxWidth: "100%", maxHeight: "80vh" }}
        />
      )}
      {isPDF && (
        <a
          href={selectedImage}
          target="_blank"
          rel="noopener noreferrer"
          style={{
            marginTop: "1rem",
            fontSize: "14px",
            textDecoration: "underline",
            color: "#007bff",
          }}
        >
          Open PDF in New Tab
        </a>
      )}
    </Modal>
  );
};

const RecordsTable = ({ filteredRecords, setSelectedImage }) => (
  <Table>
    <TableHead>
      <TableRow>
        <TableCell>Login</TableCell>
        <TableCell>Date</TableCell>
        <TableCell>Category</TableCell>
        <TableCell>Description</TableCell>
        <TableCell>Amount</TableCell>
        <TableCell>Image</TableCell>
      </TableRow>
    </TableHead>
    <TableBody>
      {filteredRecords.map((record) => (
        <TableRow key={record.id || record.category}>
          <TableCell>{record.login}</TableCell>
          <TableCell>{record.date}</TableCell>
          <TableCell>{record.category}</TableCell>
          <TableCell>
            <pre style={{ margin: 0, whiteSpace: "pre-wrap" }}>
              {record.description}
            </pre>
          </TableCell>
          <TableCell>{record.amount}</TableCell>

          <TableCell>
            {record.image && (
              <>
                {typeof record.image === "string" &&
                record.image.toLowerCase().endsWith(".pdf") ? (
                  <a
                    href={record.image}
                    target="_blank"
                    rel="noopener noreferrer"
                  >
                    View PDF
                  </a>
                ) : (
                  <div style={{ display: "inline-block" }}>
                    {" "}
                    {/* Added wrapper div */}
                    <Image
                      src={record.image}
                      alt={
                        record.image?.pathname?.toLowerCase().endsWith(".pdf")
                          ? "Click to view pdf"
                          : ""
                      }
                      style={{ width: 100, cursor: "pointer" }}
                      onClick={() => setSelectedImage(record.image)}
                      onError={(e) => {
                        e.target.style.display = "none"; // Only hide the image
                        e.target.onerror = null;
                      }} // Set src to an empty string in case of an error
                    />
                    {record.image?.pathname?.toLowerCase().endsWith(".pdf") && (
                      <span
                        style={{
                          display: "block",
                          cursor: "pointer",
                          color: "blue",
                          textDecoration: "underline",
                        }}
                        onClick={() => window.open(record.image, "_blank")}
                      >
                        Click to view pdf
                      </span>
                    )}
                  </div>
                )}
              </>
            )}
          </TableCell>
        </TableRow>
      ))}
    </TableBody>
  </Table>
);

const SearchForm = ({ searchQuery, setSearchQuery, handleSearch }) => (
  <View as="form" onSubmit={handleSearch}>
    <Flex direction="row" gap="1rem">
      <TextField
        placeholder="description or $category"
        label="Search"
        labelHidden
        variation="quiet"
        value={searchQuery}
        onChange={(e) => setSearchQuery(e.target.value)}
      />

      <Button className="btn btn-large btn-primary" type="submit" width="50%">
        Search
      </Button>
    </Flex>
  </View>
);

const CreateRecordForm = ({
  createNote,
  loginId,
  setUploadedFileName,
  uploadedFileName,
}) => (
  <View as="form" onSubmit={createNote} width="100%">
    <Flex direction="column" gap="2rem" padding="2rem" width="100%">
      <TextField
        name="login"
        defaultValue={loginId}
        placeholder="Record Login"
        label="Record Login"
        labelHidden
        variation="quiet"
        style={{ display: "none" }}
      />
      <TextField
        name="type"
        defaultValue="journal"
        placeholder="Record type"
        label="Record type"
        labelHidden
        variation="quiet"
        style={{ display: "none" }}
      />
      <TextField
        name="date"
        defaultValue={new Date(new Date() - 5 * 60 * 60 * 1000)
          .toISOString()
          .substring(0, 19)}
        placeholder="Record Date"
        label="Record Date"
        labelHidden
        variation="quiet"
      />
      <TextField
        name="category"
        placeholder="Record Category"
        label="Record Category"
        labelHidden
        variation="quiet"
      />
      <textarea
        name="description"
        placeholder="Record Description"
        label="Record Description"
        labelHidden
        variation="quiet"
        rows={4}
        required
      />
      <TextField
        name="amount"
        placeholder="Record Amount"
        label="Record Amount"
        labelHidden
        variation="quiet"
        required
      />
      <View>
        <View
          name="image"
          as="input"
          type="file"
          alignSelf="end"
          accept="image/png, image/jpeg, application/pdf"
          onChange={(e) => {
            const file = e.target.files[0];
            if (file) setUploadedFileName(file.name);
            else setUploadedFileName("");
          }}
        />
        {uploadedFileName && (
          <p style={{ marginTop: "0.5rem", fontSize: "14px", color: "#555" }}>
            Selected File: <strong>{uploadedFileName}</strong>
          </p>
        )}
      </View>
      <Button
        className="btn btn-large btn-primary"
        type="submit"
        width="50%"
        margin="2rem auto 0"
      >
        Create Record
      </Button>
    </Flex>
  </View>
);

export default function Journal() {
  const [records, setRecords] = useState([]);
  const [filteredRecords, setFilteredRecords] = useState([]);
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedImage, setSelectedImage] = useState(null);
  const [uploadedFileName, setUploadedFileName] = useState("");
  const [loginId, setLoginId] = useState(null);
  const [totalAmount, setTotalAmount] = useState(0);

  useEffect(() => {
    (async function fetchUserInfo() {
      try {
        const user = await getCurrentUser();
        const loginId = user.signInDetails?.loginId;
        setLoginId(loginId);
      } catch (error) {
        console.error(error);
      }
    })();
  }, []);

  const fetchNotes = useCallback(async () => {
    try {
      const { data: records } = await client.models.Docs.list();
      const filteredRecords = await Promise.all(
        records
          .filter((record) => true)
          .map(async (record) => {
            if (record.image) {
              const linkToStorageFile = await getUrl({
                path: `media/user_folder/${record.image}`,
              });
              record.image = linkToStorageFile.url;
            }
            return record;
          })
      );
      setRecords(records);
      setFilteredRecords(filteredRecords);
      return filteredRecords;
    } catch (error) {
      console.error("Error fetching notes:", error);
      return [];
    }
  }, []);

  const createNote = async (event) => {
    event.preventDefault();
    const form = new FormData(event.target);
    const originalImageName = form.get("image").name;
    const randomSuffix = Math.floor(Math.random() * 1000000);
    const uniqueImageName = `${Date.now()}-${randomSuffix}-${originalImageName}`;

    try {
      const { data: newNote } = await client.models.Docs.create({
        login: form.get("login"),
        type: form.get("type"),
        date: form.get("date"),
        category: form.get("category"),
        description: form.get("description"),
        amount: form.get("amount"),
        image: uniqueImageName,
      });

      if (newNote.image) {
        await uploadData({
          path: `media/user_folder/${uniqueImageName}`,
          data: form.get("image"),
        });
      }

      event.target.reset();
      setUploadedFileName("");
    } catch (error) {
      console.error("Error creating note:", error);
    }
  };

  const filterRecords = useCallback(
    (query, recordsToFilter = records) => {
      const isCategoryFilter = query.startsWith("$");
      const filtered = recordsToFilter.filter(
        (record) =>
          record.type === "journal" &&
          (isCategoryFilter
            ? record.category
                .toLowerCase()
                .includes(query.substring(1).toLowerCase())
            : record.description.toLowerCase().includes(query.toLowerCase()))
      );
      const sorted = filtered.sort(
        (a, b) => new Date(b.date) - new Date(a.date)
      );
      setFilteredRecords(sorted);
      setTotalAmount(
        filtered.reduce((sum, record) => sum + parseFloat(record.amount), 0)
      );
    },
    [records]
  );

  const handleSearch = async (event) => {
    event.preventDefault();
    try {
      const updatedRecords = await fetchNotes();
      filterRecords(searchQuery, updatedRecords);
    } catch (error) {
      console.error("Error during search:", error);
    }
  };

  return (
    <Authenticator hideSignUp={true}>
      {({ signOut }) => (
        <AppLayout>
          <Heading level={1}>Journal</Heading>
          <CreateRecordForm
            createNote={createNote}
            loginId={loginId}
            setUploadedFileName={setUploadedFileName}
            uploadedFileName={uploadedFileName}
          />
          <Divider />
          <SearchForm
            searchQuery={searchQuery}
            setSearchQuery={setSearchQuery}
            handleSearch={handleSearch}
          />
          <Heading level={2}>Records</Heading>
          <div style={{ overflowX: "auto" }}>
            <Heading level={6}>Total Amount: ${totalAmount.toFixed(2)}</Heading>
            <RecordsTable
              filteredRecords={filteredRecords}
              setSelectedImage={setSelectedImage}
            />
          </div>
          {selectedImage && (
            <ModalViewer
              selectedImage={selectedImage}
              closeModal={() => setSelectedImage(null)}
            />
          )}

          <Button onClick={signOut}>Sign Out</Button>
          {/* Footer with dynamic copyright */}
        </AppLayout>
      )}
    </Authenticator>
  );
}
